<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Services\CryptoService;
use Illuminate\Http\Request;
use App\Models\CryptoAsset;
use App\Models\Transaction;
use App\Models\CardTransaction;
use Illuminate\Support\Facades\Auth;

class CryptoController extends Controller
{
    private $cryptoService;

    public function __construct(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    public function details($symbol, $network = 'native')
{
    $user = Auth::user();
    $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
    
    // Get standard coin ID
    $coinId = $this->cryptoService->getStandardSymbol($symbol);
    
    // Get which_crypto field for transactions
    $which_crypto = strtolower($symbol);
    if (strtolower($symbol) === 'usdt') {
        $which_crypto = 'usdt_' . strtolower($network);
    }

    // Fetch transactions for this crypto, including both sides of swaps
    $transactions = Transaction::where('user_id', $user->id)
        ->where(function($query) use ($which_crypto) {
            $query->where('which_crypto', $which_crypto)
                  ->orWhere('from_crypto', $which_crypto)
                  ->orWhere('to_crypto', $which_crypto);
        })
        ->orderBy('created_at', 'desc')
        ->take(10)
        ->get();
    
    // Default asset data
    $assetData = [
        'symbol' => strtoupper($symbol),
        'balance' => 0,
        'value' => 0,
        'icon_url' => $this->cryptoService->getIconUrl($coinId),
        'network' => $network !== 'native' ? strtoupper($network) : null,
        'network_url' => $network !== 'native' ? $this->cryptoService->getNetworkIcon($network) : null,
    ];
    
    try {
        $prices = $this->cryptoService->getPrices();
        
        if ($cryptoAsset) {
            $balance = $this->getAssetBalance($cryptoAsset, $symbol, $network);
            $price = $prices[$coinId]['usd'] ?? 0;
            
            $assetData['balance'] = $balance;
            $assetData['price'] = $price;
            $assetData['value'] = $balance * $price;
            $assetData['price_change_24h'] = $prices[$coinId]['usd_24h_change'] ?? 0;
            
            // Update network URL for USDT if needed
            if (strtolower($symbol) === 'usdt' && $network !== 'native') {
                $networkSymbol = match(strtolower($network)) {
                    'trc20' => 'tron',
                    'bep20', 'bnb' => 'binancecoin',
                    'erc20' => 'ethereum',
                    default => null
                };
                $assetData['network_url'] = $networkSymbol ? $this->cryptoService->getIconUrl($networkSymbol) : null;
            }
        }
    } catch (\Exception $e) {
        logger()->error('Error fetching crypto data: ' . $e->getMessage());
    }

    return view('user.crypto.details', [
        'asset' => $assetData,
        'coinId' => $coinId,
        'transactions' => $transactions,
        'currentCrypto' => $which_crypto // Pass the current crypto identifier to the view
    ]);
}

    private function getAssetBalance(CryptoAsset $cryptoAsset, $symbol, $network)
    {
        if (strtolower($symbol) === 'usdt') {
            return match(strtolower($network)) {
                'trc20' => $cryptoAsset->usdt_trc20_balance,
                'bep20', 'bnb' => $cryptoAsset->usdt_bep20_balance,
                'erc20' => $cryptoAsset->usdt_erc20_balance,
                default => 0
            };
        }
        
        $balanceField = strtolower($symbol) . '_balance';
        return $cryptoAsset->$balanceField ?? 0;
    }

   public function manage()
{
    // Ensure user is authenticated
    if (!auth()->check()) {
        return redirect()->route('login');
    }

    // Get the user's crypto asset
    $cryptoAsset = CryptoAsset::firstOrCreate(
        ['user_id' => auth()->id()]
    );

    // Add this line to log status fields
    \Log::info('Status Fields Debug', [
        'status_fields' => $cryptoAsset->debugStatusFields()
    ]);

    // Get supported cryptocurrencies
    $supportedCryptos = CryptoAsset::getSupportedCryptos();
    
    // Prepare crypto data for the view
    $cryptos = array_map(function($key, $crypto) use ($cryptoAsset) {
        $statusField = $crypto['status_field'];
        $addressField = $crypto['address_field'];
        
        return [
            'symbol' => strtoupper($key),
            'name' => $crypto['name'],
            'icon' => $crypto['icon'],
            'color' => $crypto['color'],
            'status' => $cryptoAsset->$statusField ?? false,
            'address' => $cryptoAsset->$addressField ?? null
        ];
    }, array_keys($supportedCryptos), $supportedCryptos);

    return view('user.crypto.manage', [
        'cryptos' => $cryptos
    ]);
}

    public function updateManage(Request $request)
{
    // Validate the request
    $validated = $request->validate([
        'crypto' => 'required|string',
        'status' => 'required|boolean'
    ]);

    try {
        // Get the authenticated user
        $user = auth()->user();

        // Find or create the CryptoAsset for the user
        $cryptoAsset = CryptoAsset::firstOrCreate(
            ['user_id' => $user->id]
        );

        // Map of cryptocurrency symbols to their status fields
        $statusFieldMap = [
            'btc' => 'btc_status',
            'usdt_trc20' => 'usdt_trc20_status',
            'usdt_erc20' => 'usdt_erc20_status',
            'eth' => 'eth_status',
            'trx' => 'trx_status',
            'usdt_bep20' => 'usdt_bep20_status',
            'bch' => 'bch_status',
            'ltc' => 'ltc_status',
            'xlm' => 'xlm_status',
            'dash' => 'dash_status',
            'sol' => 'sol_status',
            'bnb' => 'bnb_status',
            'dot' => 'dot_status'
        ];

        // Validate the cryptocurrency symbol
        $cryptoSymbol = strtolower($validated['crypto']);
        if (!isset($statusFieldMap[$cryptoSymbol])) {
            return response()->json([
                'success' => false,
                'message' => 'Unsupported cryptocurrency'
            ], 400);
        }

        // Get the correct status field
        $statusField = $statusFieldMap[$cryptoSymbol];

        // Update the status
        $cryptoAsset->$statusField = $validated['status'];
        
        // Generate address if status is being enabled and no address exists
        $addressField = $cryptoSymbol . '_address';
        if ($validated['status'] && empty($cryptoAsset->$addressField)) {
            // Dynamically call the address generation method
            $generateMethod = 'generate' . ucfirst(str_replace('_', '', $cryptoSymbol)) . 'Address';
            
            if (method_exists($cryptoAsset, $generateMethod)) {
                $cryptoAsset->$addressField = $cryptoAsset->$generateMethod();
            }
        }

        // Save the changes
        $cryptoAsset->save();

        return response()->json([
            'success' => true,
            'message' => 'Cryptocurrency status updated successfully',
            'crypto' => $cryptoSymbol,
            'status' => $validated['status']
        ]);

    } catch (\Exception $e) {
        // Log the error
        \Log::error('Crypto Manage Update Error: ' . $e->getMessage());

        return response()->json([
            'success' => false,
            'message' => 'An error occurred while updating cryptocurrency status',
            'error' => $e->getMessage()
        ], 500);
    }
}

    public function address()
    {
        $user = Auth::user();
        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();

        if (!$cryptoAsset) {
            $cryptoAsset = CryptoAsset::create(['user_id' => $user->id]);
            $cryptoAsset->generateAddressesIfNull();
        } else {
            $cryptoAsset->generateAddressesIfNull();
        }

        $activeCryptos = $cryptoAsset->getActiveCryptos();

        return view('user.crypto.address', [
            'activeCryptos' => $activeCryptos,
        ]);
    }

}